/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Arrays;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.BlockOption;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.OptionSet;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.eclipse.californium.core.network.stack.BlockwiseStatus;

public class BlockwiseLayer
extends AbstractLayer {
    private static final Logger LOGGER = Logger.getLogger(BlockwiseLayer.class.getName());
    private int maxMessageSize;
    private int preferredBlockSize;
    private int blockTimeout;
    private int maxResourceBodySize;

    public BlockwiseLayer(NetworkConfig config) {
        this.maxMessageSize = config.getInt("MAX_MESSAGE_SIZE", 1024);
        this.preferredBlockSize = config.getInt("PREFERRED_BLOCK_SIZE", 512);
        this.blockTimeout = config.getInt("BLOCKWISE_STATUS_LIFETIME");
        this.maxResourceBodySize = config.getInt("MAX_RESOURCE_BODY_SIZE", 2048);
        LOGGER.log(Level.CONFIG, "BlockwiseLayer uses MAX_MESSAGE_SIZE={0}, DEFAULT_BLOCK_SIZE={1}, BLOCKWISE_STATUS_LIFETIME={2} and MAX_RESOURCE_BODY_SIZE={3}", new Object[]{this.maxMessageSize, this.preferredBlockSize, this.blockTimeout, this.maxResourceBodySize});
    }

    @Override
    public void sendRequest(Exchange exchange, Request request) {
        BlockOption block2 = request.getOptions().getBlock2();
        if (block2 != null && block2.getNum() > 0) {
            LOGGER.fine("request contains block2 option, creating random-access blockwise status");
            BlockwiseStatus status = new BlockwiseStatus(BlockwiseLayer.getSizeForSzx(block2.getSzx()), request.getOptions().getContentFormat());
            status.setCurrentSzx(block2.getSzx());
            status.setCurrentNum(block2.getNum());
            status.setRandomAccess(true);
            exchange.setResponseBlockStatus(status);
            this.lower().sendRequest(exchange, request);
        } else if (this.requiresBlockwise(request)) {
            this.startBlockwiseUpload(exchange, request);
        } else {
            exchange.setCurrentRequest(request);
            this.lower().sendRequest(exchange, request);
        }
    }

    private void startBlockwiseUpload(Exchange exchange, Request request) {
        BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
        Request block = BlockwiseLayer.getNextRequestBlock(request, status);
        block.getOptions().setSize1(request.getPayloadSize());
        exchange.setRequestBlockStatus(status);
        exchange.setCurrentRequest(block);
        this.lower().sendRequest(exchange, block);
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        BlockOption block1 = request.getOptions().getBlock1();
        if (block1 != null) {
            LOGGER.log(Level.FINE, "inbound request contains block1 option {0}", block1);
            if (this.isTransparentBlockwiseHandlingEnabled()) {
                this.handleInboundBlockwiseUpload(block1, exchange, request);
            } else {
                LOGGER.fine("transparent blockwise handling is disabled, delivering request to application layer");
                this.upper().receiveRequest(exchange, request);
            }
        } else if (exchange.getResponse() != null && request.getOptions().hasBlock2()) {
            BlockOption block2 = request.getOptions().getBlock2();
            Response response = exchange.getResponse();
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            status.setCurrentNum(block2.getNum());
            status.setCurrentSzx(block2.getSzx());
            Response block = BlockwiseLayer.getNextResponseBlock(response, status);
            block.getOptions().setSize2(response.getPayloadSize());
            if (status.isComplete()) {
                LOGGER.log(Level.FINE, "peer has requested last block of blockwise transfer: {0}", status);
                exchange.setResponseBlockStatus(null);
                exchange.setBlockCleanupHandle(null);
            } else {
                LOGGER.log(Level.FINE, "peer has requested intermediary block of blockwise transfer: {0}", status);
            }
            exchange.setCurrentResponse(block);
            this.lower().sendResponse(exchange, block);
        } else {
            BlockwiseLayer.earlyBlock2Negotiation(exchange, request);
            exchange.setRequest(request);
            this.upper().receiveRequest(exchange, request);
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private void handleInboundBlockwiseUpload(BlockOption block1, Exchange exchange, Request request) {
        if (this.requestExceedsMaxBodySize(request)) {
            Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_TOO_LARGE);
            error.setPayload(String.format("body too large, can process %d bytes max", this.maxResourceBodySize));
            error.getOptions().setSize1(this.maxResourceBodySize);
            this.lower().sendResponse(exchange, error);
            return;
        }
        BlockwiseStatus status = this.findRequestBlockStatus(exchange, request);
        if (block1.getNum() == 0 && status.getCurrentNum() > 0) {
            LOGGER.finer("Block1 num is 0, the client has restarted the blockwise transfer. Reset status.");
            exchange.setRequestBlockStatus(null);
            status = this.findRequestBlockStatus(exchange, request);
        }
        if (block1.getNum() != status.getCurrentNum()) {
            LOGGER.log(Level.WARNING, "Wrong block number. Expected {0} but received {1}. Respond with 4.08 (Request Entity Incomplete)", new Object[]{status.getCurrentNum(), block1.getNum()});
            Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
            error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
            error.setPayload("Wrong block number");
            exchange.setCurrentResponse(error);
            this.lower().sendResponse(exchange, error);
            return;
        }
        if (!status.hasContentFormat(request.getOptions().getContentFormat())) {
            Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE);
            error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
            error.setPayload("unexpected Content-Format");
            exchange.setCurrentResponse(error);
            this.lower().sendResponse(exchange, error);
            return;
        }
        status.addBlock(request.getPayload());
        status.setCurrentNum(status.getCurrentNum() + 1);
        if (block1.isM()) {
            LOGGER.finest("There are more blocks to come. Acknowledge this block.");
            Response piggybacked = Response.createResponse(request, CoAP.ResponseCode.CONTINUE);
            piggybacked.getOptions().setBlock1(block1.getSzx(), true, block1.getNum());
            piggybacked.setLast(false);
            exchange.setCurrentResponse(piggybacked);
            this.lower().sendResponse(exchange, piggybacked);
            return;
        }
        LOGGER.finer("This was the last block. Deliver request");
        exchange.setBlock1ToAck(block1);
        BlockwiseLayer.earlyBlock2Negotiation(exchange, request);
        Request assembled = new Request(request.getCode());
        assembled.setSenderIdentity(request.getSenderIdentity());
        BlockwiseLayer.assembleMessage(status, assembled);
        exchange.setRequest(assembled);
        this.upper().receiveRequest(exchange, assembled);
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        BlockOption block1 = exchange.getBlock1ToAck();
        if (block1 != null) {
            exchange.setBlock1ToAck(null);
        }
        if (this.requiresBlockwise(exchange, response)) {
            BlockwiseStatus status = this.findResponseBlockStatus(exchange, response);
            int bodySize = response.getPayloadSize();
            Response block = BlockwiseLayer.getNextResponseBlock(response, status);
            block.getOptions().setSize2(bodySize);
            if (block1 != null) {
                block.getOptions().setBlock1(block1);
            }
            if (status.isComplete()) {
                LOGGER.log(Level.FINE, "Ongoing finished on first block {0}", status);
                exchange.setResponseBlockStatus(null);
                exchange.setBlockCleanupHandle(null);
            } else {
                LOGGER.log(Level.FINE, "Ongoing started {0}", status);
            }
            exchange.setCurrentResponse(block);
            this.lower().sendResponse(exchange, block);
        } else {
            if (block1 != null) {
                response.getOptions().setBlock1(block1);
            }
            exchange.setCurrentResponse(response);
            exchange.setBlockCleanupHandle(null);
            this.lower().sendResponse(exchange, response);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        if (exchange.getRequest().isCanceled()) {
            if (response.getType() != CoAP.Type.ACK) {
                LOGGER.finer("rejecting blockwise transfer for canceled Exchange");
                EmptyMessage rst = EmptyMessage.newRST(response);
                this.sendEmptyMessage(exchange, rst);
            }
        } else if (!response.hasBlockOption()) {
            exchange.setResponse(response);
            this.upper().receiveResponse(exchange, response);
        } else {
            BlockOption block = response.getOptions().getBlock1();
            if (block != null) {
                this.handleBlock1Response(exchange, response, block);
            }
            if ((block = response.getOptions().getBlock2()) != null) {
                this.handleBlock2Response(exchange, response, block);
            }
        }
    }

    private void handleBlock1Response(Exchange exchange, Response response, BlockOption block1) {
        LOGGER.log(Level.FINER, "received response acknowledging block {0}", block1);
        BlockwiseStatus status = exchange.getRequestBlockStatus();
        if (status == null) {
            LOGGER.log(Level.FINE, "discarding response containing unexpected block1 option: {0}", response);
        } else if (!status.isComplete()) {
            if (block1.isM()) {
                this.sendNextBlock(exchange, response, block1, status);
            } else {
                this.sendNextBlock(exchange, response, block1, status);
            }
        } else if (!response.getOptions().hasBlock2()) {
            this.upper().receiveResponse(exchange, response);
        } else {
            LOGGER.finer("Block1 followed by Block2 transfer");
        }
    }

    private void sendNextBlock(Exchange exchange, Response response, BlockOption block1, BlockwiseStatus requestStatus) {
        int newSzx;
        int newSize;
        int currentSize = 1 << 4 + requestStatus.getCurrentSzx();
        if (block1.getSize() < currentSize) {
            newSize = block1.getSize();
            newSzx = block1.getSzx();
        } else {
            newSize = currentSize;
            newSzx = requestStatus.getCurrentSzx();
        }
        int nextNum = requestStatus.getCurrentNum() + currentSize / newSize;
        LOGGER.log(Level.FINER, "Sending next Block1 num={0}", nextNum);
        requestStatus.setCurrentNum(nextNum);
        requestStatus.setCurrentSzx(newSzx);
        Request nextBlock = BlockwiseLayer.getNextRequestBlock(exchange.getRequest(), requestStatus);
        nextBlock.getOptions().setSize1(exchange.getRequest().getPayloadSize());
        nextBlock.setToken(response.getToken());
        exchange.setCurrentRequest(nextBlock);
        this.lower().sendRequest(exchange, nextBlock);
    }

    private void handleBlock2Response(Exchange exchange, Response response, BlockOption block2) {
        if (this.responseExceedsMaxBodySize(response)) {
            LOGGER.log(Level.FINE, "requested resource body exceeds max buffer size [{0}], aborting request", this.maxResourceBodySize);
            exchange.getRequest().cancel();
            return;
        }
        BlockwiseStatus responseStatus = this.findResponseBlockStatus(exchange, response);
        if (response.isNotification() && block2.getNum() == 0 && responseStatus.getCurrentNum() != 0) {
            if (response.getOptions().getObserve() > responseStatus.getObserve()) {
                LOGGER.log(Level.WARNING, "ongoing blockwise transfer reset at num = {0} by new notification: {1}", new Object[]{responseStatus.getCurrentNum(), response});
                exchange.setResponseBlockStatus(null);
                responseStatus = this.findResponseBlockStatus(exchange, response);
            } else {
                LOGGER.log(Level.FINE, "discarding old notification received during ongoing blockwise transfer: {0}", response);
                return;
            }
        }
        if (block2.getNum() == responseStatus.getCurrentNum() && (block2.getNum() == 0 || Arrays.equals(response.getToken(), exchange.getCurrentRequest().getToken()))) {
            if (!responseStatus.addBlock(response.getPayload())) {
                LOGGER.log(Level.FINE, "requested resource body exceeds max buffer size [{0}], aborting request", this.maxResourceBodySize);
                exchange.getRequest().cancel();
                return;
            }
            if (response.getOptions().hasObserve()) {
                responseStatus.setObserve(response.getOptions().getObserve());
            }
            if (responseStatus.isRandomAccess()) {
                exchange.setResponse(response);
                this.upper().receiveResponse(exchange, response);
            } else if (block2.isM()) {
                Request request = exchange.getRequest();
                int num = block2.getNum() + 1;
                int szx = block2.getSzx();
                boolean m = false;
                LOGGER.log(Level.FINER, "Requesting next Block2 num={0}", num);
                Request block = new Request(request.getCode());
                block.setType(request.getType());
                block.setDestination(request.getDestination());
                block.setDestinationPort(request.getDestinationPort());
                if (!response.getOptions().hasObserve()) {
                    block.setToken(response.getToken());
                }
                block.setOptions(new OptionSet(request.getOptions()));
                block.getOptions().removeObserve();
                block.getOptions().setBlock2(szx, m, num);
                block.addMessageObservers(request.getMessageObservers());
                responseStatus.setCurrentNum(num);
                exchange.setCurrentRequest(block);
                this.lower().sendRequest(exchange, block);
            } else {
                LOGGER.log(Level.FINER, "We have received all {0} blocks of the response. Assemble and deliver", responseStatus.getBlockCount());
                Response assembled = new Response(response.getCode());
                BlockwiseLayer.assembleMessage(responseStatus, assembled);
                assembled.setRTT(System.currentTimeMillis() - exchange.getTimestamp());
                int observe = responseStatus.getObserve();
                if (observe != -1) {
                    if (!response.getOptions().hasObserve()) {
                        exchange.completeCurrentRequest();
                    }
                    assembled.getOptions().setObserve(observe);
                    exchange.setResponseBlockStatus(null);
                }
                LOGGER.log(Level.FINE, "Assembled response: {0}", assembled);
                exchange.setResponse(assembled);
                this.upper().receiveResponse(exchange, assembled);
            }
        } else {
            LOGGER.log(Level.WARNING, "Wrong block number. Expected {0} but received {1}: {2}", new Object[]{responseStatus.getCurrentNum(), block2.getNum(), response});
            if (response.getType() == CoAP.Type.CON) {
                EmptyMessage rst = EmptyMessage.newRST(response);
                this.lower().sendEmptyMessage(exchange, rst);
            }
        }
    }

    private static void earlyBlock2Negotiation(Exchange exchange, Request request) {
        BlockOption block2 = request.getOptions().getBlock2();
        if (block2 != null) {
            BlockwiseStatus status2 = new BlockwiseStatus(request.getOptions().getContentFormat(), block2.getNum(), block2.getSzx());
            LOGGER.log(Level.FINE, "Request with early block negotiation {0}. Create and set new Block2 status: {1}", new Object[]{block2, status2});
            exchange.setResponseBlockStatus(status2);
        }
    }

    private BlockwiseStatus findRequestBlockStatus(Exchange exchange, Request request) {
        BlockwiseStatus status = exchange.getRequestBlockStatus();
        if (status == null) {
            if (exchange.isOfLocalOrigin()) {
                status = new BlockwiseStatus(this.preferredBlockSize, request.getOptions().getContentFormat());
            } else {
                int bufferSize = this.maxResourceBodySize;
                if (request.getOptions().hasBlock1() && request.getOptions().hasSize1()) {
                    bufferSize = request.getOptions().getSize1();
                }
                status = new BlockwiseStatus(bufferSize, request.getOptions().getContentFormat());
            }
            status.setFirst(request);
            status.setCurrentSzx(BlockwiseLayer.computeSZX(this.preferredBlockSize));
            exchange.setRequestBlockStatus(status);
            LOGGER.log(Level.FINER, "There is no assembler status yet. Create and set new Block1 status: {0}", status);
        } else {
            LOGGER.log(Level.FINER, "Current Block1 status: {0}", status);
        }
        this.prepareBlockCleanup(exchange);
        return status;
    }

    private BlockwiseStatus findResponseBlockStatus(Exchange exchange, Response response) {
        BlockwiseStatus status = exchange.getResponseBlockStatus();
        if (status == null) {
            if (exchange.isOfLocalOrigin()) {
                int bufferSize = this.maxResourceBodySize;
                if (response.getOptions().hasBlock2() && response.getOptions().hasSize2()) {
                    bufferSize = response.getOptions().getSize2();
                }
                status = new BlockwiseStatus(bufferSize, response.getOptions().getContentFormat());
            } else {
                status = new BlockwiseStatus(0, response.getOptions().getContentFormat());
            }
            status.setCurrentSzx(BlockwiseLayer.computeSZX(this.preferredBlockSize));
            status.setFirst(response);
            exchange.setResponseBlockStatus(status);
            LOGGER.log(Level.FINER, "There is no blockwise status yet. Create and set new Block2 status: {0}", status);
        } else {
            LOGGER.log(Level.FINER, "Current Block2 status: {0}", status);
        }
        this.prepareBlockCleanup(exchange);
        return status;
    }

    private static Request getNextRequestBlock(Request request, BlockwiseStatus status) {
        int num = status.getCurrentNum();
        int szx = status.getCurrentSzx();
        Request block = new Request(request.getCode());
        block.setType(request.getType());
        block.setDestination(request.getDestination());
        block.setDestinationPort(request.getDestinationPort());
        block.setOptions(new OptionSet(request.getOptions()));
        block.addMessageObservers(request.getMessageObservers());
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        int to = Math.min((num + 1) * currentSize, request.getPayloadSize());
        int length = to - from;
        byte[] blockPayload = new byte[length];
        System.arraycopy(request.getPayload(), from, blockPayload, 0, length);
        block.setPayload(blockPayload);
        boolean m = to < request.getPayloadSize();
        block.getOptions().setBlock1(szx, m, num);
        status.setComplete(!m);
        return block;
    }

    private static Response getNextResponseBlock(Response response, BlockwiseStatus status) {
        Response block;
        int szx = status.getCurrentSzx();
        int num = status.getCurrentNum();
        if (response.getOptions().hasObserve()) {
            block = response;
        } else {
            block = new Response(response.getCode());
            block.setDestination(response.getDestination());
            block.setDestinationPort(response.getDestinationPort());
            block.setOptions(new OptionSet(response.getOptions()));
            block.addMessageObserver(new TimeoutForwarder(response));
        }
        int payloadsize = response.getPayloadSize();
        int currentSize = 1 << 4 + szx;
        int from = num * currentSize;
        if (payloadsize > 0 && from < payloadsize) {
            int to = Math.min((num + 1) * currentSize, response.getPayloadSize());
            int length = to - from;
            byte[] blockPayload = new byte[length];
            boolean m = to < response.getPayloadSize();
            block.getOptions().setBlock2(szx, m, num);
            System.arraycopy(response.getPayload(), from, blockPayload, 0, length);
            block.setPayload(blockPayload);
            block.setLast(!m && !response.getOptions().hasObserve());
            status.setComplete(!m);
        } else {
            block.getOptions().setBlock2(szx, false, num);
            block.setLast(true);
            status.setComplete(true);
        }
        return block;
    }

    private static void assembleMessage(BlockwiseStatus status, Message message) {
        message.setSource(status.getFirst().getSource());
        message.setSourcePort(status.getFirst().getSourcePort());
        message.setType(status.getFirst().getType());
        message.setMID(status.getFirst().getMID());
        message.setToken(status.getFirst().getToken());
        message.setOptions(new OptionSet(status.getFirst().getOptions()));
        message.setPayload(status.getBody());
    }

    private boolean requiresBlockwise(Request request) {
        boolean blockwiseRequired = false;
        if (request.getCode() == CoAP.Code.PUT || request.getCode() == CoAP.Code.POST) {
            boolean bl = blockwiseRequired = request.getPayloadSize() > this.maxMessageSize;
        }
        if (blockwiseRequired) {
            LOGGER.log(Level.FINE, "request body [{0}/{1}] requires blockwise trasnfer", new Object[]{request.getPayloadSize(), this.maxMessageSize});
        }
        return blockwiseRequired;
    }

    private boolean requiresBlockwise(Exchange exchange, Response response) {
        boolean blockwiseRequired;
        boolean bl = blockwiseRequired = response.getPayloadSize() > this.maxMessageSize || exchange.getResponseBlockStatus() != null;
        if (blockwiseRequired) {
            LOGGER.log(Level.FINE, "response body [{0}/{1}] requires blockwise transfer", new Object[]{response.getPayloadSize(), this.maxMessageSize});
        }
        return blockwiseRequired;
    }

    private boolean isTransparentBlockwiseHandlingEnabled() {
        return this.maxResourceBodySize > 0;
    }

    private boolean responseExceedsMaxBodySize(Response response) {
        return response.getOptions().hasSize2() && response.getOptions().getSize2() > this.maxResourceBodySize;
    }

    private boolean requestExceedsMaxBodySize(Request request) {
        return request.getOptions().hasSize1() && request.getOptions().getSize1() > this.maxResourceBodySize;
    }

    static int computeSZX(int blockSize) {
        if (blockSize > 1024) {
            return 6;
        }
        if (blockSize <= 16) {
            return 0;
        }
        int maxOneBit = Integer.highestOneBit(blockSize);
        return Integer.numberOfTrailingZeros(maxOneBit) - 4;
    }

    static int getSizeForSzx(int szx) {
        if (szx <= 0) {
            return 16;
        }
        if (szx >= 6) {
            return 1024;
        }
        return 1 << szx + 4;
    }

    protected void prepareBlockCleanup(Exchange exchange) {
        if (this.executor.isShutdown()) {
            LOGGER.info("Endpoint is being destroyed: skipping block clean-up");
            return;
        }
        BlockCleanupTask task = new BlockCleanupTask(exchange);
        ScheduledFuture<?> f = this.executor.schedule(task, (long)this.blockTimeout, TimeUnit.MILLISECONDS);
        exchange.setBlockCleanupHandle(f);
    }

    protected class BlockCleanupTask
    implements Runnable {
        private final Exchange exchange;

        public BlockCleanupTask(Exchange exchange) {
            this.exchange = exchange;
        }

        @Override
        public void run() {
            if (this.exchange.getRequest() == null) {
                LOGGER.log(Level.INFO, "Block1 transfer timed out: {0}", this.exchange.getCurrentRequest());
            } else {
                LOGGER.log(Level.INFO, "Block2 transfer timed out: {0}", this.exchange.getRequest());
            }
            this.exchange.setComplete();
        }
    }

    public static class TimeoutForwarder
    extends MessageObserverAdapter {
        private final Message message;

        public TimeoutForwarder(Message message) {
            this.message = message;
        }

        @Override
        public void onTimeout() {
            this.message.setTimedOut(true);
        }
    }
}

