/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.net.InetSocketAddress;
import java.security.SecureRandom;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.Utils;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.InMemoryMessageIdProvider;
import org.eclipse.californium.core.network.InMemoryRandomTokenProvider;
import org.eclipse.californium.core.network.MessageExchangeStore;
import org.eclipse.californium.core.network.MessageIdProvider;
import org.eclipse.californium.core.network.TokenProvider;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.deduplication.Deduplicator;
import org.eclipse.californium.core.network.deduplication.DeduplicatorFactory;
import org.eclipse.californium.elements.CorrelationContext;

public class InMemoryMessageExchangeStore
implements MessageExchangeStore {
    private static final Logger LOGGER = Logger.getLogger(InMemoryMessageExchangeStore.class.getName());
    private final ConcurrentMap<Exchange.KeyMID, Exchange> exchangesByMID = new ConcurrentHashMap<Exchange.KeyMID, Exchange>();
    private final ConcurrentMap<Exchange.KeyToken, Exchange> exchangesByToken = new ConcurrentHashMap<Exchange.KeyToken, Exchange>();
    private final ConcurrentMap<Exchange.KeyUri, Exchange> ongoingExchanges = new ConcurrentHashMap<Exchange.KeyUri, Exchange>();
    private final NetworkConfig config;
    private boolean running = false;
    private Deduplicator deduplicator;
    private ScheduledFuture<?> statusLogger;
    private ScheduledExecutorService scheduler;
    private MessageIdProvider messageIdProvider;
    private TokenProvider tokenProvider;
    private SecureRandom secureRandom;

    public InMemoryMessageExchangeStore(NetworkConfig config) {
        this(config, new InMemoryRandomTokenProvider(config));
        LOGGER.log(Level.CONFIG, "using default TokenProvider {0}", InMemoryRandomTokenProvider.class.getName());
    }

    public InMemoryMessageExchangeStore(NetworkConfig config, TokenProvider tokenProvider) {
        if (config == null) {
            throw new NullPointerException("Configuration must not be null");
        }
        if (tokenProvider == null) {
            throw new NullPointerException("TokenProvider must not be null");
        }
        this.tokenProvider = tokenProvider;
        this.config = config;
    }

    private void startStatusLogging() {
        final Level healthStatusLevel = Level.parse(this.config.getString("HEALTH_STATUS_PRINT_LEVEL", Level.FINEST.getName()));
        int healthStatusInterval = this.config.getInt("HEALTH_STATUS_INTERVAL", 60);
        if (LOGGER.isLoggable(healthStatusLevel)) {
            this.scheduler = Executors.newSingleThreadScheduledExecutor(new Utils.DaemonThreadFactory("MessageExchangeStore"));
            this.statusLogger = this.scheduler.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    LOGGER.log(healthStatusLevel, InMemoryMessageExchangeStore.this.dumpCurrentLoadLevels());
                }
            }, healthStatusInterval, healthStatusInterval, TimeUnit.SECONDS);
        }
    }

    private String dumpCurrentLoadLevels() {
        StringBuilder b = new StringBuilder("MessageExchangeStore contents: ");
        b.append(this.exchangesByMID.size()).append(" exchanges by MID, ");
        b.append(this.exchangesByToken.size()).append(" exchanges by token, ");
        b.append(this.ongoingExchanges.size()).append(" ongoing blockwise exchanges");
        return b.toString();
    }

    public synchronized void setDeduplicator(Deduplicator deduplicator) {
        if (this.running) {
            throw new IllegalStateException("Cannot set Deduplicator when store is already started");
        }
        if (deduplicator == null) {
            throw new NullPointerException("Deduplicator must not be null");
        }
        this.deduplicator = deduplicator;
    }

    public synchronized void setMessageIdProvider(MessageIdProvider provider) {
        if (this.running) {
            throw new IllegalStateException("Cannot set messageIdProvider when store is already started");
        }
        if (provider == null) {
            throw new NullPointerException("Message ID Provider must not be null");
        }
        this.messageIdProvider = provider;
    }

    @Override
    public boolean isEmpty() {
        LOGGER.finer(this.dumpCurrentLoadLevels());
        return this.exchangesByMID.isEmpty() && this.exchangesByToken.isEmpty() && this.ongoingExchanges.isEmpty() && this.deduplicator.isEmpty();
    }

    @Override
    public void assignMessageId(Message message) {
        if (message.getMID() == -1) {
            InetSocketAddress dest = new InetSocketAddress(message.getDestination(), message.getDestinationPort());
            int mid = this.messageIdProvider.getNextMessageId(dest);
            if (mid < 0) {
                LOGGER.log(Level.WARNING, "Cannot send message to {0}, all MIDs are in use", dest);
            } else {
                message.setMID(mid);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerWithMessageId(Exchange exchange, Message message) {
        MessageIdProvider messageIdProvider = this.messageIdProvider;
        synchronized (messageIdProvider) {
            if (message.getMID() == -1) {
                InetSocketAddress dest = new InetSocketAddress(message.getDestination(), message.getDestinationPort());
                int mid = this.messageIdProvider.getNextMessageId(dest);
                if (mid < 0) {
                    LOGGER.log(Level.WARNING, "Cannot send message to {0}, all MIDs are in use", dest);
                } else {
                    message.setMID(mid);
                    Exchange.KeyMID key = Exchange.KeyMID.fromOutboundMessage(message);
                    if (this.exchangesByMID.putIfAbsent(key, exchange) != null) {
                        LOGGER.log(Level.WARNING, "newly generated MID [{0}] already in use, overwriting already registered exchange", message.getMID());
                    }
                }
            } else {
                Exchange existingExchange = this.exchangesByMID.putIfAbsent(Exchange.KeyMID.fromOutboundMessage(message), exchange);
                if (existingExchange != null) {
                    if (existingExchange != exchange) {
                        throw new IllegalArgumentException(String.format("message ID [%d] already in use, cannot register exchange", message.getMID()));
                    }
                    if (exchange.getFailedTransmissionCount() == 0) {
                        throw new IllegalArgumentException(String.format("message with already registered ID [%d] is not a re-transmission, cannot register exchange", message.getMID()));
                    }
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerWithToken(Exchange exchange) {
        Request request = exchange.getCurrentRequest();
        ConcurrentMap<Exchange.KeyToken, Exchange> concurrentMap = this.exchangesByToken;
        synchronized (concurrentMap) {
            Exchange.KeyToken idByToken;
            if (request.getToken() == null) {
                idByToken = this.tokenProvider.getUnusedToken(request);
                request.setToken(idByToken.getToken());
            } else {
                idByToken = Exchange.KeyToken.fromOutboundMessage(request);
                if (exchange.getFailedTransmissionCount() <= 0 && !request.getOptions().hasBlock1() && !request.getOptions().hasBlock2() && !request.getOptions().hasObserve() && this.tokenProvider.isTokenInUse(idByToken)) {
                    LOGGER.log(Level.WARNING, "Manual token overrides existing open request: {0}", idByToken);
                }
            }
            this.exchangesByToken.put(idByToken, exchange);
        }
    }

    @Override
    public void registerOutboundRequest(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentRequest() == null) {
            throw new IllegalArgumentException("exchange does not contain a request");
        }
        this.registerWithMessageId(exchange, exchange.getCurrentRequest());
        this.registerWithToken(exchange);
    }

    @Override
    public void registerOutboundRequestWithTokenOnly(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentRequest() == null) {
            throw new IllegalArgumentException("exchange does not contain a request");
        }
        this.registerWithToken(exchange);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void remove(Exchange.KeyToken token) {
        ConcurrentMap<Exchange.KeyToken, Exchange> concurrentMap = this.exchangesByToken;
        synchronized (concurrentMap) {
            this.exchangesByToken.remove(token);
            LOGGER.log(Level.FINE, "removing exchange for token {0}, remaining exchanges by tokens: {1}", new Object[]{token, this.exchangesByToken.size()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Exchange remove(Exchange.KeyMID messageId) {
        MessageIdProvider messageIdProvider = this.messageIdProvider;
        synchronized (messageIdProvider) {
            Exchange removedExchange = (Exchange)this.exchangesByMID.remove(messageId);
            LOGGER.log(Level.FINE, "removing exchange for MID {0}, remaining exchanges by MIDs: {1}", new Object[]{messageId, this.exchangesByMID.size()});
            return removedExchange;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Exchange get(Exchange.KeyToken token) {
        if (token == null) {
            return null;
        }
        ConcurrentMap<Exchange.KeyToken, Exchange> concurrentMap = this.exchangesByToken;
        synchronized (concurrentMap) {
            return (Exchange)this.exchangesByToken.get(token);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Exchange get(Exchange.KeyMID messageId) {
        if (messageId == null) {
            return null;
        }
        MessageIdProvider messageIdProvider = this.messageIdProvider;
        synchronized (messageIdProvider) {
            return (Exchange)this.exchangesByMID.get(messageId);
        }
    }

    @Override
    public void setContext(Exchange.KeyToken token, CorrelationContext correlationContext) {
    }

    @Override
    public void registerOutboundResponse(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentResponse() == null) {
            throw new IllegalArgumentException("exchange does not contain a response");
        }
        this.registerWithMessageId(exchange, exchange.getCurrentResponse());
    }

    @Override
    public Exchange get(Exchange.KeyUri requestUri) {
        if (requestUri == null) {
            return null;
        }
        return (Exchange)this.ongoingExchanges.get(requestUri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() {
        MessageIdProvider messageIdProvider = this.messageIdProvider;
        synchronized (messageIdProvider) {
            ConcurrentMap<Exchange.KeyToken, Exchange> concurrentMap = this.exchangesByToken;
            synchronized (concurrentMap) {
                this.exchangesByMID.clear();
                this.exchangesByToken.clear();
                this.ongoingExchanges.clear();
            }
        }
    }

    @Override
    public Exchange registerBlockwiseExchange(Exchange.KeyUri requestUri, Exchange exchange) {
        return this.ongoingExchanges.put(requestUri, exchange);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void remove(Exchange.KeyUri requestUri) {
        ConcurrentMap<Exchange.KeyUri, Exchange> concurrentMap = this.ongoingExchanges;
        synchronized (concurrentMap) {
            this.ongoingExchanges.remove(requestUri);
            LOGGER.log(Level.FINE, "removing transfer for URI {0}, remaining ongoing exchanges: {1}", new Object[]{requestUri, this.ongoingExchanges.size()});
        }
    }

    @Override
    public synchronized void start() {
        if (!this.running) {
            this.startStatusLogging();
            if (this.deduplicator == null) {
                DeduplicatorFactory factory = DeduplicatorFactory.getDeduplicatorFactory();
                this.deduplicator = factory.createDeduplicator(this.config);
            }
            this.deduplicator.start();
            if (this.messageIdProvider == null) {
                LOGGER.log(Level.CONFIG, "no MessageIdProvider set, using default {0}", InMemoryMessageIdProvider.class.getName());
                this.messageIdProvider = new InMemoryMessageIdProvider(this.config);
            }
            this.secureRandom = new SecureRandom();
            this.secureRandom.nextInt(10);
            this.running = true;
        }
    }

    @Override
    public synchronized void stop() {
        if (this.running) {
            if (this.statusLogger != null) {
                this.statusLogger.cancel(false);
            }
            this.deduplicator.stop();
            this.clear();
            this.running = false;
        }
    }

    @Override
    public Exchange findPrevious(Exchange.KeyMID messageId, Exchange exchange) {
        return this.deduplicator.findPrevious(messageId, exchange);
    }

    @Override
    public Exchange find(Exchange.KeyMID messageId) {
        return this.deduplicator.find(messageId);
    }

    @Override
    public void releaseToken(Exchange.KeyToken keyToken) {
        this.tokenProvider.releaseToken(keyToken);
    }
}

